/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/**
 * \file    nandflash_test.c
 *
 * \brief   Support routines to test the NAND Flash
 *
 * This file contains the suppoort routines for testing the NAND 
 * Flash on the DM388 EVM
 *
 * \author  0043
 *
 * \version 01a,25aug,2010 Created
 */

/*
 *====================
 * Includes
 *====================
 */
#include <stdio.h>
#include "dm388_EVM.h"
#include "dm388_types.h"
#include "gpio.h"
#include "nandflash.h"

#undef SPARE_CHECK

int set_gpio1_bank1 (int gpio, int value)
{
	unsigned int regval;
	static int initialised = 0;

	if (!initialised) {
	/* Reset GPIO Subsystem */
		GPIO1_SYSCONFIG = 0x00000020; /* Software Reset */
		DM388_usecDelay(1000);
		GPIO1_SYSCONFIG = 0x100; /* no-idle */
		initialised = 1;
	}

	/* Output Enable GPIO16 in Bank 2 */
	regval = GPIO1_OE;
	regval = regval & ~(1 << gpio);
	GPIO1_OE = regval;

	if (!value) {
		/* Set WLAN_ENABLE Low */
		regval = GPIO1_DATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO1_DATAOUT = regval;
		regval = GPIO1_SETDATAOUT;
		regval = regval & ~(1 << gpio);
		GPIO1_SETDATAOUT = regval;
	} else {
		/* Set BT_ENABLE High */
		GPIO1_DATAOUT |= (1 <<  gpio);
		GPIO1_SETDATAOUT |= (1 << gpio);
	}

	return 1;
}

/** **************************************************************************
 * \n \brief Routine to Test Nandflash device .
 *
 * This routine tests first 4 blocks of device without Spare area.Uncommenting 
 * the  loop tests with spare area.This routine first erases the blocks.Writes 
 * into the device and read the same and Compares both.
 * 
 *
 * \return
 * \n      return 0 for success  
 * \n      return 1 for error   
 */

INT32 nandflash_test(void *testargs)
{
    platform_write("\n********************************\n\r");
    platform_write(  "         Nandflash Test       \n\r");
    platform_write(  "********************************\n\r");
	INT32 u32RetVal = SUCCESS;
    INT16 status;
    UINT32 i, j, errors, blockstotest,wpstatus;

	static UINT16 src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE];
	static UINT16 dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE];

    /* Initialize NAND */
	//NANDInit();

	/* Disabling Write protect */
	/* A arbitary huge delay to make sure gpio is pulled high */
	set_gpio1_bank1 (DM388_GPMC_WPn, 1);
	DM388_usecDelay(1000);
	set_gpio1_bank1 (DM388_GPMC_WPn, 0);
	DM388_usecDelay(1000);
	set_gpio1_bank1 (DM388_GPMC_WPn, 1);
	DM388_usecDelay(1000); /* settle down */

	platform_write("\nRunning NAND Flash Chip Detect Test\n");
    u32RetVal = nand_init();
    if(FAILED == u32RetVal)
    {
    	u32RetVal = FAILED;
        return u32RetVal;
    }
    else
    {
    	platform_write("NAND Flash Chip Detect Test Passed\n");
    }
    
    platform_write("\nRunning NAND Flash Block Erase Test\n");
    blockstotest = NANDFLASH_BLOCKS_TEST;  /* NANDFLASH_BLOCKS */
    platform_write("Testing %d blocks\n", blockstotest);

    /* Erase all of NAND */
    platform_write("Erasing NAND blocks\n");
    errors = 0;
    for (i = 0; i < blockstotest ; i++)
    {
    	u32RetVal = nand_eraseBlock(i);
        if (FAILED == u32RetVal)
		{
            platform_write(" -- block %d erase error\n", i);
            errors++;
        }
    }
    
    platform_write("--> %d block erase errors\n", errors);
    if (errors > 80)
    {
    	platform_write ("Too many block erase failed. Aborting test.!!\r\n");
        return u32RetVal ;
    }

    if(u32RetVal != FAILED)
        {
          	platform_write("NAND Flash Block Erase Test Passed\n");
        }
    platform_write("\nRunning NAND Flash Memory Access Test\n");
    /* Fill the source buffer */
#ifdef SPARE_CHECK
    for (i = 0; i < (NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE); i++)
#else
    for (i = 0; i < NANDFLASH_PAGESIZE; i++)
#endif
     src[i] = 0xAAAA;

    /* Clear the destination buffer */
#ifdef SPARE_CHECK
    for (i = 0; i < (NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE); i++)
#else
    for (i = 0; i < NANDFLASH_PAGESIZE; i++)
#endif
        dst[i] = 0x0000;
	    
    /* Program all pages */
    platform_write("Programming NAND pages\n");
    for (j = 0; j < (blockstotest * NANDFLASH_PAGESPERBLOCK); j++)
    {
        src[0] = j & 0x000000ff;
        src[1] = (j & 0x0000ff00) >> 8;
        src[2] = (j & 0x00ff0000) >> 16;
        src[3] = (j & 0xff000000) >> 24;

#ifdef SPARE_CHECK
	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 4] =  j & 0x000000ff;
	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 3] = (j & 0x0000ff00) >> 8;
	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 2] = (j & 0x00ff0000) >> 16;
	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 1] = (j & 0xff000000) >> 24;
#endif

#ifdef SPARE_CHECK
	nand_writePage(j, src, NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE);
#else
    nand_writePage(j, src, NANDFLASH_PAGESIZE);
#endif
    }

    /* Read and compare all pages */
    platform_write("Comparing data\n");
    errors = 0;
    for (j = 0; j < (blockstotest * NANDFLASH_PAGESPERBLOCK); j++)
    {
#ifdef SPARE_CHECK
    nand_readPage(j, dst, NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE);
#else
    nand_readPage(j, dst, NANDFLASH_PAGESIZE);
#endif
    status = 0;
    if (dst[0] != (j & 0x000000ff))
        status = 1;
    if (dst[1] != ((j & 0x0000ff00) >> 8))
        status = 1;
    if (dst[2] != ((j & 0x00ff0000) >> 16))
        status = 1;
    if (dst[3] != ((j & 0xff000000) >> 24))
        status = 1;

#ifdef SPARE_CHECK
	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 1] != ((j & 0xff000000) >> 24))
	    status = 1;
	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 2] != ((j & 0x00ff0000) >> 16))
	    status = 1;
	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 3] != ((j & 0x0000ff00) >> 8))
	    status = 1;
	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 4] != (j & 0x000000ff))
	    status = 1;
#endif

	if (status)
	  {
            errors++;
	  }
	if (status)
	  {
	    platform_write("Error on page %d\n", j);
      }
    }

    platform_write("--> %d page verify errors\n", errors);
    if (errors > 0)  // 5120 for entire device
    {
	platform_write("NAND Read/write failed\n");
	platform_write("NAND tests failed!!\n");
	return 1;
    }
    else
    {
    platform_write("NAND Flash Memory Access Test Passed\n");
    }

    platform_write("\nRunning NAND Flash Write Protect Test\n");
    /* Enabling Write protect */
    /* A arbitary huge delay to make sure gpio is pulled low*/
    set_gpio1_bank1 (DM388_GPMC_WPn, 0);
    DM388_usecDelay(1000);
    set_gpio1_bank1 (DM388_GPMC_WPn, 1);
    DM388_usecDelay(1000);
    set_gpio1_bank1 (DM388_GPMC_WPn, 0);
    DM388_usecDelay(1000); /* settle down */

    // Check erase status
    NANDFLASH_CLE = CMD_STATUS;
    wpstatus = NANDFLASH_DATA;
    if(wpstatus & 0x80)
    {
    	platform_write("Write protect not enabled!!\n");
    }

    /* Fill the source buffer */
#ifdef SPARE_CHECK
    for (i = 0; i < (NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE); i++)
#else
    for (i = 0; i < NANDFLASH_PAGESIZE; i++)
#endif
     src[i] = 0x2222;

    /* Clear the destination buffer */
#ifdef SPARE_CHECK
    for (i = 0; i < (NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE); i++)
#else
    for (i = 0; i < NANDFLASH_PAGESIZE; i++)
#endif
        dst[i] = 0x0000;

    /* Program all pages */
    platform_write("Testing NAND write protection\n");

    j = 0;
    src[0] = j & 0x000000ff;
    src[1] = (j & 0x0000ff00) >> 8;
    src[2] = (j & 0x00ff0000) >> 16;
    src[3] = (j & 0xff000000) >> 24;

    #ifdef SPARE_CHECK
    	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 4] =  j & 0x000000ff;
    	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 3] = (j & 0x0000ff00) >> 8;
    	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 2] = (j & 0x00ff0000) >> 16;
    	src[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 1] = (j & 0xff000000) >> 24;
    #endif

    #ifdef SPARE_CHECK
    	nand_writePage(j, src, NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE);
    #else
        nand_writePage(j, src, NANDFLASH_PAGESIZE);
    #endif

    /* Read and compare page after write protect*/
    platform_write("Checking to see if data has been protected\n");
    errors = 0;
    j = 0;
    #ifdef SPARE_CHECK
        nand_readPage(j, dst, NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE);
    #else
        nand_readPage(j, dst, NANDFLASH_PAGESIZE);
    #endif
    status = 0;
    if (dst[0] != (j & 0x000000ff))
        status = 1;
    if (dst[1] != ((j & 0x0000ff00) >> 8))
        status = 1;
    if (dst[2] != ((j & 0x00ff0000) >> 16))
        status = 1;
    if (dst[3] != ((j & 0xff000000) >> 24))
        status = 1;
    for (j = 4; j < 1024; j++)
    {
    if (dst[j] != 0xAAAA)
    	status = 1;
    }

    #ifdef SPARE_CHECK
    	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 1] != ((j & 0xff000000) >> 24))
    	    status = 1;
    	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 2] != ((j & 0x00ff0000) >> 16))
    	    status = 1;
    	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 3] != ((j & 0x0000ff00) >> 8))
    	    status = 1;
    	if (dst[NANDFLASH_PAGESIZE + NANDFLASH_SPARESIZE - 4] != (j & 0x000000ff))
    	    status = 1;
    #endif

    if (status)
    {
      	platform_write("Write protection failed! %d\n",status);
       	platform_write("NAND tests failed!!\n");
       	return 1;
    }
    else
    {
    	platform_write("NAND Flash Write Protect Test Passed\n");
       	platform_write("\nNand Flash Passed!\n");
       	platform_write("\nNand Flash Tests Completed!!\n");
       	platform_write("-----------x----------");
       	return 0;
    }
}
